// Copyright 2012-2015 The Rust Project Developers. See the COPYRIGHT
// file at the top-level directory of this distribution and at
// http://rust-lang.org/COPYRIGHT.
//
// Licensed under the Apache License, Version 2.0 <LICENSE-APACHE or
// http://www.apache.org/licenses/LICENSE-2.0> or the MIT license
// <LICENSE-MIT or http://opensource.org/licenses/MIT>, at your
// option. This file may not be copied, modified, or distributed
// except according to those terms.

//! The implementation of the query system itself. Defines the macros
//! that generate the actual methods on tcx which find and execute the
//! provider, manage the caches, and so forth.

use dep_graph::{DepNodeIndex, DepNode, DepKind};
use errors::{Diagnostic, DiagnosticBuilder};
use ty::{TyCtxt};
use ty::maps::Query; // NB: actually generated by the macros in this file
use ty::maps::config::QueryDescription;
use ty::item_path;

use rustc_data_structures::fx::{FxHashMap};
use std::cell::{RefMut, Cell};
use std::marker::PhantomData;
use std::mem;
use syntax_pos::Span;

pub(super) struct QueryMap<D: QueryDescription> {
    phantom: PhantomData<D>,
    pub(super) map: FxHashMap<D::Key, QueryValue<D::Value>>,
}

pub(super) struct QueryValue<T> {
    pub(super) value: T,
    pub(super) index: DepNodeIndex,
    pub(super) diagnostics: Option<Box<QueryDiagnostics>>,
}

impl<T> QueryValue<T> {
    pub(super) fn new(value: T,
                      dep_node_index: DepNodeIndex,
                      diagnostics: Vec<Diagnostic>)
                      -> QueryValue<T> {
        QueryValue {
            value,
            index: dep_node_index,
            diagnostics: if diagnostics.len() == 0 {
                None
            } else {
                Some(Box::new(QueryDiagnostics {
                    diagnostics,
                    emitted_diagnostics: Cell::new(true),
                }))
            },
        }
    }
}

pub(super) struct QueryDiagnostics {
    pub(super) diagnostics: Vec<Diagnostic>,
    pub(super) emitted_diagnostics: Cell<bool>,
}

impl<M: QueryDescription> QueryMap<M> {
    pub(super) fn new() -> QueryMap<M> {
        QueryMap {
            phantom: PhantomData,
            map: FxHashMap(),
        }
    }
}

pub(super) struct CycleError<'a, 'tcx: 'a> {
    span: Span,
    cycle: RefMut<'a, [(Span, Query<'tcx>)]>,
}

impl<'a, 'gcx, 'tcx> TyCtxt<'a, 'gcx, 'tcx> {
    pub(super) fn report_cycle(self, CycleError { span, cycle }: CycleError)
        -> DiagnosticBuilder<'a>
    {
        // Subtle: release the refcell lock before invoking `describe()`
        // below by dropping `cycle`.
        let stack = cycle.to_vec();
        mem::drop(cycle);

        assert!(!stack.is_empty());

        // Disable naming impls with types in this path, since that
        // sometimes cycles itself, leading to extra cycle errors.
        // (And cycle errors around impls tend to occur during the
        // collect/coherence phases anyhow.)
        item_path::with_forced_impl_filename_line(|| {
            let mut err =
                struct_span_err!(self.sess, span, E0391,
                                 "unsupported cyclic reference between types/traits detected");
            err.span_label(span, "cyclic reference");

            err.span_note(stack[0].0, &format!("the cycle begins when {}...",
                                               stack[0].1.describe(self)));

            for &(span, ref query) in &stack[1..] {
                err.span_note(span, &format!("...which then requires {}...",
                                             query.describe(self)));
            }

            err.note(&format!("...which then again requires {}, completing the cycle.",
                              stack[0].1.describe(self)));

            return err
        })
    }

    pub(super) fn cycle_check<F, R>(self, span: Span, query: Query<'gcx>, compute: F)
                                    -> Result<R, CycleError<'a, 'gcx>>
        where F: FnOnce() -> R
    {
        {
            let mut stack = self.maps.query_stack.borrow_mut();
            if let Some((i, _)) = stack.iter().enumerate().rev()
                                       .find(|&(_, &(_, ref q))| *q == query) {
                return Err(CycleError {
                    span,
                    cycle: RefMut::map(stack, |stack| &mut stack[i..])
                });
            }
            stack.push((span, query));
        }

        let result = compute();

        self.maps.query_stack.borrow_mut().pop();

        Ok(result)
    }
}

// If enabled, send a message to the profile-queries thread
macro_rules! profq_msg {
    ($tcx:expr, $msg:expr) => {
        if cfg!(debug_assertions) {
            if  $tcx.sess.profile_queries() {
                profq_msg($msg)
            }
        }
    }
}

// If enabled, format a key using its debug string, which can be
// expensive to compute (in terms of time).
macro_rules! profq_key {
    ($tcx:expr, $key:expr) => {
        if cfg!(debug_assertions) {
            if $tcx.sess.profile_queries_and_keys() {
                Some(format!("{:?}", $key))
            } else { None }
        } else { None }
    }
}

macro_rules! define_maps {
    (<$tcx:tt>
     $($(#[$attr:meta])*
       [$($modifiers:tt)*] fn $name:ident: $node:ident($K:ty) -> $V:ty,)*) => {

        use dep_graph::DepNodeIndex;
        use std::cell::RefCell;

        define_map_struct! {
            tcx: $tcx,
            input: ($(([$($modifiers)*] [$($attr)*] [$name]))*)
        }

        impl<$tcx> Maps<$tcx> {
            pub fn new(providers: IndexVec<CrateNum, Providers<$tcx>>)
                       -> Self {
                Maps {
                    providers,
                    query_stack: RefCell::new(vec![]),
                    $($name: RefCell::new(QueryMap::new())),*
                }
            }
        }

        #[allow(bad_style)]
        #[derive(Copy, Clone, Debug, PartialEq, Eq)]
        pub enum Query<$tcx> {
            $($(#[$attr])* $name($K)),*
        }

        #[allow(bad_style)]
        #[derive(Clone, Debug, PartialEq, Eq)]
        pub enum QueryMsg {
            $($name(Option<String>)),*
        }

        impl<$tcx> Query<$tcx> {
            pub fn describe(&self, tcx: TyCtxt) -> String {
                let (r, name) = match *self {
                    $(Query::$name(key) => {
                        (queries::$name::describe(tcx, key), stringify!($name))
                    })*
                };
                if tcx.sess.verbose() {
                    format!("{} [{}]", r, name)
                } else {
                    r
                }
            }
        }

        pub mod queries {
            use std::marker::PhantomData;

            $(#[allow(bad_style)]
            pub struct $name<$tcx> {
                data: PhantomData<&$tcx ()>
            })*
        }

        $(impl<$tcx> QueryConfig for queries::$name<$tcx> {
            type Key = $K;
            type Value = $V;
        }

        impl<'a, $tcx, 'lcx> queries::$name<$tcx> {

            #[allow(unused)]
            fn to_dep_node(tcx: TyCtxt<'a, $tcx, 'lcx>, key: &$K) -> DepNode {
                use dep_graph::DepConstructor::*;

                DepNode::new(tcx, $node(*key))
            }

            fn try_get_with(tcx: TyCtxt<'a, $tcx, 'lcx>,
                            mut span: Span,
                            key: $K)
                            -> Result<$V, CycleError<'a, $tcx>>
            {
                debug!("ty::queries::{}::try_get_with(key={:?}, span={:?})",
                       stringify!($name),
                       key,
                       span);

                profq_msg!(tcx,
                    ProfileQueriesMsg::QueryBegin(
                        span.data(),
                        QueryMsg::$name(profq_key!(tcx, key))
                    )
                );

                if let Some(value) = tcx.maps.$name.borrow().map.get(&key) {
                    if let Some(ref d) = value.diagnostics {
                        if !d.emitted_diagnostics.get() {
                            d.emitted_diagnostics.set(true);
                            let handle = tcx.sess.diagnostic();
                            for diagnostic in d.diagnostics.iter() {
                                DiagnosticBuilder::new_diagnostic(handle, diagnostic.clone())
                                    .emit();
                            }
                        }
                    }
                    profq_msg!(tcx, ProfileQueriesMsg::CacheHit);
                    tcx.dep_graph.read_index(value.index);
                    return Ok((&value.value).clone());
                }

                // FIXME(eddyb) Get more valid Span's on queries.
                // def_span guard is necessary to prevent a recursive loop,
                // default_span calls def_span query internally.
                if span == DUMMY_SP && stringify!($name) != "def_span" {
                    span = key.default_span(tcx)
                }

                // Fast path for when incr. comp. is off. `to_dep_node` is
                // expensive for some DepKinds.
                if !tcx.dep_graph.is_fully_enabled() {
                    let null_dep_node = DepNode::new_no_params(::dep_graph::DepKind::Null);
                    return Self::force(tcx, key, span, null_dep_node)
                                .map(|(v, _)| v);
                }

                let dep_node = Self::to_dep_node(tcx, &key);

                if dep_node.kind.is_anon() {
                    profq_msg!(tcx, ProfileQueriesMsg::ProviderBegin);

                    let res = tcx.cycle_check(span, Query::$name(key), || {
                        tcx.sess.diagnostic().track_diagnostics(|| {
                            tcx.dep_graph.with_anon_task(dep_node.kind, || {
                                Self::compute_result(tcx.global_tcx(), key)
                            })
                        })
                    })?;

                    profq_msg!(tcx, ProfileQueriesMsg::ProviderEnd);
                    let ((result, dep_node_index), diagnostics) = res;

                    tcx.dep_graph.read_index(dep_node_index);
                    let value = QueryValue::new(result, dep_node_index, diagnostics);

                    return Ok((&tcx.maps
                                    .$name
                                    .borrow_mut()
                                    .map
                                    .entry(key)
                                    .or_insert(value)
                                    .value).clone());
                }

                if !dep_node.kind.is_input() {
                    use dep_graph::DepNodeColor;
                    if let Some(DepNodeColor::Green(dep_node_index)) = tcx.dep_graph
                                                                          .node_color(&dep_node) {
                        profq_msg!(tcx, ProfileQueriesMsg::CacheHit);
                        tcx.dep_graph.read_index(dep_node_index);
                        return Self::load_from_disk_and_cache_in_memory(tcx,
                                                                        key,
                                                                        span,
                                                                        dep_node_index)
                    }

                    debug!("ty::queries::{}::try_get_with(key={:?}) - running try_mark_green",
                           stringify!($name),
                           key);

                    if let Some(dep_node_index) = tcx.dep_graph.try_mark_green(tcx, &dep_node) {
                        debug_assert!(tcx.dep_graph.is_green(dep_node_index));
                        profq_msg!(tcx, ProfileQueriesMsg::CacheHit);
                        tcx.dep_graph.read_index(dep_node_index);
                        return Self::load_from_disk_and_cache_in_memory(tcx,
                                                                        key,
                                                                        span,
                                                                        dep_node_index)
                    }
                }

                match Self::force(tcx, key, span, dep_node) {
                    Ok((result, dep_node_index)) => {
                        tcx.dep_graph.read_index(dep_node_index);
                        Ok(result)
                    }
                    Err(e) => Err(e)
                }
            }

            fn compute_result(tcx: TyCtxt<'a, $tcx, 'lcx>, key: $K) -> $V {
                let provider = tcx.maps.providers[key.map_crate()].$name;
                provider(tcx.global_tcx(), key)
            }

            fn load_from_disk_and_cache_in_memory(tcx: TyCtxt<'a, $tcx, 'lcx>,
                                                  key: $K,
                                                  span: Span,
                                                  dep_node_index: DepNodeIndex)
                                                  -> Result<$V, CycleError<'a, $tcx>>
            {
                debug_assert!(tcx.dep_graph.is_green(dep_node_index));

                // We don't do any caching yet, so recompute
                let (result, diagnostics) = tcx.cycle_check(span, Query::$name(key), || {
                    tcx.sess.diagnostic().track_diagnostics(|| {
                        // The dep-graph for this computation is already in place
                        tcx.dep_graph.with_ignore(|| {
                            Self::compute_result(tcx, key)
                        })
                    })
                })?;

                if tcx.sess.opts.debugging_opts.query_dep_graph {
                    tcx.dep_graph.mark_loaded_from_cache(dep_node_index, true);
                }

                let value = QueryValue::new(result, dep_node_index, diagnostics);

                Ok((&tcx.maps
                         .$name
                         .borrow_mut()
                         .map
                         .entry(key)
                         .or_insert(value)
                         .value).clone())
            }

            fn force(tcx: TyCtxt<'a, $tcx, 'lcx>,
                     key: $K,
                     span: Span,
                     dep_node: DepNode)
                     -> Result<($V, DepNodeIndex), CycleError<'a, $tcx>> {
                debug_assert!(tcx.dep_graph.node_color(&dep_node).is_none());

                profq_msg!(tcx, ProfileQueriesMsg::ProviderBegin);
                let res = tcx.cycle_check(span, Query::$name(key), || {
                    tcx.sess.diagnostic().track_diagnostics(|| {
                        tcx.dep_graph.with_task(dep_node,
                                                tcx,
                                                key,
                                                Self::compute_result)
                    })
                })?;
                profq_msg!(tcx, ProfileQueriesMsg::ProviderEnd);

                let ((result, dep_node_index), diagnostics) = res;

                if tcx.sess.opts.debugging_opts.query_dep_graph {
                    tcx.dep_graph.mark_loaded_from_cache(dep_node_index, false);
                }

                let value = QueryValue::new(result, dep_node_index, diagnostics);

                Ok(((&tcx.maps
                         .$name
                         .borrow_mut()
                         .map
                         .entry(key)
                         .or_insert(value)
                         .value).clone(),
                   dep_node_index))
            }

            pub fn try_get(tcx: TyCtxt<'a, $tcx, 'lcx>, span: Span, key: $K)
                           -> Result<$V, DiagnosticBuilder<'a>> {
                match Self::try_get_with(tcx, span, key) {
                    Ok(e) => Ok(e),
                    Err(e) => Err(tcx.report_cycle(e)),
                }
            }
        })*

        #[derive(Copy, Clone)]
        pub struct TyCtxtAt<'a, 'gcx: 'a+'tcx, 'tcx: 'a> {
            pub tcx: TyCtxt<'a, 'gcx, 'tcx>,
            pub span: Span,
        }

        impl<'a, 'gcx, 'tcx> Deref for TyCtxtAt<'a, 'gcx, 'tcx> {
            type Target = TyCtxt<'a, 'gcx, 'tcx>;
            fn deref(&self) -> &Self::Target {
                &self.tcx
            }
        }

        impl<'a, $tcx, 'lcx> TyCtxt<'a, $tcx, 'lcx> {
            /// Return a transparent wrapper for `TyCtxt` which uses
            /// `span` as the location of queries performed through it.
            pub fn at(self, span: Span) -> TyCtxtAt<'a, $tcx, 'lcx> {
                TyCtxtAt {
                    tcx: self,
                    span
                }
            }

            $($(#[$attr])*
            pub fn $name(self, key: $K) -> $V {
                self.at(DUMMY_SP).$name(key)
            })*
        }

        impl<'a, $tcx, 'lcx> TyCtxtAt<'a, $tcx, 'lcx> {
            $($(#[$attr])*
            pub fn $name(self, key: $K) -> $V {
                queries::$name::try_get(self.tcx, self.span, key).unwrap_or_else(|mut e| {
                    e.emit();
                    Value::from_cycle_error(self.global_tcx())
                })
            })*
        }

        define_provider_struct! {
            tcx: $tcx,
            input: ($(([$($modifiers)*] [$name] [$K] [$V]))*),
            output: ()
        }

        impl<$tcx> Copy for Providers<$tcx> {}
        impl<$tcx> Clone for Providers<$tcx> {
            fn clone(&self) -> Self { *self }
        }
    }
}

macro_rules! define_map_struct {
    // Initial state
    (tcx: $tcx:tt,
     input: $input:tt) => {
        define_map_struct! {
            tcx: $tcx,
            input: $input,
            output: ()
        }
    };

    // Final output
    (tcx: $tcx:tt,
     input: (),
     output: ($($output:tt)*)) => {
        pub struct Maps<$tcx> {
            providers: IndexVec<CrateNum, Providers<$tcx>>,
            query_stack: RefCell<Vec<(Span, Query<$tcx>)>>,
            $($output)*
        }
    };

    // Field recognized and ready to shift into the output
    (tcx: $tcx:tt,
     ready: ([$($pub:tt)*] [$($attr:tt)*] [$name:ident]),
     input: $input:tt,
     output: ($($output:tt)*)) => {
        define_map_struct! {
            tcx: $tcx,
            input: $input,
            output: ($($output)*
                     $(#[$attr])* $($pub)* $name: RefCell<QueryMap<queries::$name<$tcx>>>,)
        }
    };

    // No modifiers left? This is a private item.
    (tcx: $tcx:tt,
     input: (([] $attrs:tt $name:tt) $($input:tt)*),
     output: $output:tt) => {
        define_map_struct! {
            tcx: $tcx,
            ready: ([] $attrs $name),
            input: ($($input)*),
            output: $output
        }
    };

    // Skip other modifiers
    (tcx: $tcx:tt,
     input: (([$other_modifier:tt $($modifiers:tt)*] $($fields:tt)*) $($input:tt)*),
     output: $output:tt) => {
        define_map_struct! {
            tcx: $tcx,
            input: (([$($modifiers)*] $($fields)*) $($input)*),
            output: $output
        }
    };
}

macro_rules! define_provider_struct {
    // Initial state:
    (tcx: $tcx:tt, input: $input:tt) => {
        define_provider_struct! {
            tcx: $tcx,
            input: $input,
            output: ()
        }
    };

    // Final state:
    (tcx: $tcx:tt,
     input: (),
     output: ($(([$name:ident] [$K:ty] [$R:ty]))*)) => {
        pub struct Providers<$tcx> {
            $(pub $name: for<'a> fn(TyCtxt<'a, $tcx, $tcx>, $K) -> $R,)*
        }

        impl<$tcx> Default for Providers<$tcx> {
            fn default() -> Self {
                $(fn $name<'a, $tcx>(_: TyCtxt<'a, $tcx, $tcx>, key: $K) -> $R {
                    bug!("tcx.maps.{}({:?}) unsupported by its crate",
                         stringify!($name), key);
                })*
                Providers { $($name),* }
            }
        }
    };

    // Something ready to shift:
    (tcx: $tcx:tt,
     ready: ($name:tt $K:tt $V:tt),
     input: $input:tt,
     output: ($($output:tt)*)) => {
        define_provider_struct! {
            tcx: $tcx,
            input: $input,
            output: ($($output)* ($name $K $V))
        }
    };

    // Regular queries produce a `V` only.
    (tcx: $tcx:tt,
     input: (([] $name:tt $K:tt $V:tt) $($input:tt)*),
     output: $output:tt) => {
        define_provider_struct! {
            tcx: $tcx,
            ready: ($name $K $V),
            input: ($($input)*),
            output: $output
        }
    };

    // Skip modifiers.
    (tcx: $tcx:tt,
     input: (([$other_modifier:tt $($modifiers:tt)*] $($fields:tt)*) $($input:tt)*),
     output: $output:tt) => {
        define_provider_struct! {
            tcx: $tcx,
            input: (([$($modifiers)*] $($fields)*) $($input)*),
            output: $output
        }
    };
}

pub fn force_from_dep_node<'a, 'gcx, 'lcx>(tcx: TyCtxt<'a, 'gcx, 'lcx>,
                                           dep_node: &DepNode)
                                           -> bool {
    use ty::maps::keys::Key;
    use hir::def_id::LOCAL_CRATE;

    // We must avoid ever having to call force_from_dep_node() for a
    // DepNode::CodegenUnit:
    // Since we cannot reconstruct the query key of a DepNode::CodegenUnit, we
    // would always end up having to evaluate the first caller of the
    // `codegen_unit` query that *is* reconstructible. This might very well be
    // the `compile_codegen_unit` query, thus re-translating the whole CGU just
    // to re-trigger calling the `codegen_unit` query with the right key. At
    // that point we would already have re-done all the work we are trying to
    // avoid doing in the first place.
    // The solution is simple: Just explicitly call the `codegen_unit` query for
    // each CGU, right after partitioning. This way `try_mark_green` will always
    // hit the cache instead of having to go through `force_from_dep_node`.
    // This assertion makes sure, we actually keep applying the solution above.
    debug_assert!(dep_node.kind != DepKind::CodegenUnit,
                  "calling force_from_dep_node() on DepKind::CodegenUnit");

    if !dep_node.kind.can_reconstruct_query_key() {
        return false
    }

    macro_rules! def_id {
        () => {
            if let Some(def_id) = dep_node.extract_def_id(tcx) {
                def_id
            } else {
                // return from the whole function
                return false
            }
        }
    };

    macro_rules! krate {
        () => { (def_id!()).krate }
    };

    macro_rules! force {
        ($query:ident, $key:expr) => {
            {
                use $crate::util::common::{ProfileQueriesMsg, profq_msg};

                // FIXME(eddyb) Get more valid Span's on queries.
                // def_span guard is necessary to prevent a recursive loop,
                // default_span calls def_span query internally.
                let span = if stringify!($query) != "def_span" {
                    $key.default_span(tcx)
                } else {
                    ::syntax_pos::DUMMY_SP
                };

                profq_msg!(tcx,
                    ProfileQueriesMsg::QueryBegin(
                        span.data(),
                        ::ty::maps::QueryMsg::$query(profq_key!(tcx, $key))
                    )
                );

                match ::ty::maps::queries::$query::force(tcx, $key, span, *dep_node) {
                    Ok(_) => {},
                    Err(e) => {
                        tcx.report_cycle(e).emit();
                    }
                }
            }
        }
    };

    // FIXME(#45015): We should try move this boilerplate code into a macro
    //                somehow.
    match dep_node.kind {
        // These are inputs that are expected to be pre-allocated and that
        // should therefore always be red or green already
        DepKind::AllLocalTraitImpls |
        DepKind::Krate |
        DepKind::CrateMetadata |
        DepKind::HirBody |
        DepKind::Hir |

        // This are anonymous nodes
        DepKind::IsCopy |
        DepKind::IsSized |
        DepKind::IsFreeze |
        DepKind::NeedsDrop |
        DepKind::Layout |
        DepKind::TraitSelect |
        DepKind::ConstEval |

        // We don't have enough information to reconstruct the query key of
        // these
        DepKind::InstanceSymbolName |
        DepKind::MirShim |
        DepKind::BorrowCheckKrate |
        DepKind::Specializes |
        DepKind::ImplementationsOfTrait |
        DepKind::TypeParamPredicates |
        DepKind::CodegenUnit |
        DepKind::CompileCodegenUnit |

        // These are just odd
        DepKind::Null |
        DepKind::WorkProduct => {
            bug!("force_from_dep_node() - Encountered {:?}", dep_node.kind)
        }

        // These are not queries
        DepKind::CoherenceCheckTrait |
        DepKind::ItemVarianceConstraints => {
            return false
        }

        DepKind::RegionScopeTree => { force!(region_scope_tree, def_id!()); }

        DepKind::Coherence => { force!(crate_inherent_impls, LOCAL_CRATE); }
        DepKind::CoherenceInherentImplOverlapCheck => {
            force!(crate_inherent_impls_overlap_check, LOCAL_CRATE)
        },
        DepKind::PrivacyAccessLevels => { force!(privacy_access_levels, LOCAL_CRATE); }
        DepKind::MirBuilt => { force!(mir_built, def_id!()); }
        DepKind::MirConstQualif => { force!(mir_const_qualif, def_id!()); }
        DepKind::MirConst => { force!(mir_const, def_id!()); }
        DepKind::MirValidated => { force!(mir_validated, def_id!()); }
        DepKind::MirOptimized => { force!(optimized_mir, def_id!()); }

        DepKind::BorrowCheck => { force!(borrowck, def_id!()); }
        DepKind::MirBorrowCheck => { force!(mir_borrowck, def_id!()); }
        DepKind::UnsafetyCheckResult => { force!(unsafety_check_result, def_id!()); }
        DepKind::Reachability => { force!(reachable_set, LOCAL_CRATE); }
        DepKind::MirKeys => { force!(mir_keys, LOCAL_CRATE); }
        DepKind::CrateVariances => { force!(crate_variances, LOCAL_CRATE); }
        DepKind::AssociatedItems => { force!(associated_item, def_id!()); }
        DepKind::TypeOfItem => { force!(type_of, def_id!()); }
        DepKind::GenericsOfItem => { force!(generics_of, def_id!()); }
        DepKind::PredicatesOfItem => { force!(predicates_of, def_id!()); }
        DepKind::SuperPredicatesOfItem => { force!(super_predicates_of, def_id!()); }
        DepKind::TraitDefOfItem => { force!(trait_def, def_id!()); }
        DepKind::AdtDefOfItem => { force!(adt_def, def_id!()); }
        DepKind::IsDefaultImpl => { force!(is_default_impl, def_id!()); }
        DepKind::ImplTraitRef => { force!(impl_trait_ref, def_id!()); }
        DepKind::ImplPolarity => { force!(impl_polarity, def_id!()); }
        DepKind::ClosureKind => { force!(closure_kind, def_id!()); }
        DepKind::FnSignature => { force!(fn_sig, def_id!()); }
        DepKind::GenSignature => { force!(generator_sig, def_id!()); }
        DepKind::CoerceUnsizedInfo => { force!(coerce_unsized_info, def_id!()); }
        DepKind::ItemVariances => { force!(variances_of, def_id!()); }
        DepKind::IsConstFn => { force!(is_const_fn, def_id!()); }
        DepKind::IsForeignItem => { force!(is_foreign_item, def_id!()); }
        DepKind::SizedConstraint => { force!(adt_sized_constraint, def_id!()); }
        DepKind::DtorckConstraint => { force!(adt_dtorck_constraint, def_id!()); }
        DepKind::AdtDestructor => { force!(adt_destructor, def_id!()); }
        DepKind::AssociatedItemDefIds => { force!(associated_item_def_ids, def_id!()); }
        DepKind::InherentImpls => { force!(inherent_impls, def_id!()); }
        DepKind::TypeckBodiesKrate => { force!(typeck_item_bodies, LOCAL_CRATE); }
        DepKind::TypeckTables => { force!(typeck_tables_of, def_id!()); }
        DepKind::HasTypeckTables => { force!(has_typeck_tables, def_id!()); }
        DepKind::SymbolName => { force!(def_symbol_name, def_id!()); }
        DepKind::SpecializationGraph => { force!(specialization_graph_of, def_id!()); }
        DepKind::ObjectSafety => { force!(is_object_safe, def_id!()); }
        DepKind::TraitImpls => { force!(trait_impls_of, def_id!()); }

        DepKind::ParamEnv => { force!(param_env, def_id!()); }
        DepKind::DescribeDef => { force!(describe_def, def_id!()); }
        DepKind::DefSpan => { force!(def_span, def_id!()); }
        DepKind::LookupStability => { force!(lookup_stability, def_id!()); }
        DepKind::LookupDeprecationEntry => {
            force!(lookup_deprecation_entry, def_id!());
        }
        DepKind::ItemBodyNestedBodies => { force!(item_body_nested_bodies, def_id!()); }
        DepKind::ConstIsRvaluePromotableToStatic => {
            force!(const_is_rvalue_promotable_to_static, def_id!());
        }
        DepKind::ImplParent => { force!(impl_parent, def_id!()); }
        DepKind::TraitOfItem => { force!(trait_of_item, def_id!()); }
        DepKind::IsExportedSymbol => { force!(is_exported_symbol, def_id!()); }
        DepKind::IsMirAvailable => { force!(is_mir_available, def_id!()); }
        DepKind::ItemAttrs => { force!(item_attrs, def_id!()); }
        DepKind::FnArgNames => { force!(fn_arg_names, def_id!()); }
        DepKind::DylibDepFormats => { force!(dylib_dependency_formats, krate!()); }
        DepKind::IsPanicRuntime => { force!(is_panic_runtime, krate!()); }
        DepKind::IsCompilerBuiltins => { force!(is_compiler_builtins, krate!()); }
        DepKind::HasGlobalAllocator => { force!(has_global_allocator, krate!()); }
        DepKind::ExternCrate => { force!(extern_crate, def_id!()); }
        DepKind::LintLevels => { force!(lint_levels, LOCAL_CRATE); }
        DepKind::InScopeTraits => { force!(in_scope_traits_map, def_id!().index); }
        DepKind::ModuleExports => { force!(module_exports, def_id!()); }
        DepKind::IsSanitizerRuntime => { force!(is_sanitizer_runtime, krate!()); }
        DepKind::IsProfilerRuntime => { force!(is_profiler_runtime, krate!()); }
        DepKind::GetPanicStrategy => { force!(panic_strategy, krate!()); }
        DepKind::IsNoBuiltins => { force!(is_no_builtins, krate!()); }
        DepKind::ImplDefaultness => { force!(impl_defaultness, def_id!()); }
        DepKind::ExportedSymbolIds => { force!(exported_symbol_ids, krate!()); }
        DepKind::NativeLibraries => { force!(native_libraries, krate!()); }
        DepKind::PluginRegistrarFn => { force!(plugin_registrar_fn, krate!()); }
        DepKind::DeriveRegistrarFn => { force!(derive_registrar_fn, krate!()); }
        DepKind::CrateDisambiguator => { force!(crate_disambiguator, krate!()); }
        DepKind::CrateHash => { force!(crate_hash, krate!()); }
        DepKind::OriginalCrateName => { force!(original_crate_name, krate!()); }

        DepKind::AllTraitImplementations => {
            force!(all_trait_implementations, krate!());
        }

        DepKind::IsDllimportForeignItem => {
            force!(is_dllimport_foreign_item, def_id!());
        }
        DepKind::IsStaticallyIncludedForeignItem => {
            force!(is_statically_included_foreign_item, def_id!());
        }
        DepKind::NativeLibraryKind => { force!(native_library_kind, def_id!()); }
        DepKind::LinkArgs => { force!(link_args, LOCAL_CRATE); }

        DepKind::NamedRegion => { force!(named_region_map, def_id!().index); }
        DepKind::IsLateBound => { force!(is_late_bound_map, def_id!().index); }
        DepKind::ObjectLifetimeDefaults => {
            force!(object_lifetime_defaults_map, def_id!().index);
        }

        DepKind::Visibility => { force!(visibility, def_id!()); }
        DepKind::DepKind => { force!(dep_kind, krate!()); }
        DepKind::CrateName => { force!(crate_name, krate!()); }
        DepKind::ItemChildren => { force!(item_children, def_id!()); }
        DepKind::ExternModStmtCnum => { force!(extern_mod_stmt_cnum, def_id!()); }
        DepKind::GetLangItems => { force!(get_lang_items, LOCAL_CRATE); }
        DepKind::DefinedLangItems => { force!(defined_lang_items, krate!()); }
        DepKind::MissingLangItems => { force!(missing_lang_items, krate!()); }
        DepKind::ExternConstBody => { force!(extern_const_body, def_id!()); }
        DepKind::VisibleParentMap => { force!(visible_parent_map, LOCAL_CRATE); }
        DepKind::MissingExternCrateItem => {
            force!(missing_extern_crate_item, krate!());
        }
        DepKind::UsedCrateSource => { force!(used_crate_source, krate!()); }
        DepKind::PostorderCnums => { force!(postorder_cnums, LOCAL_CRATE); }
        DepKind::HasCloneClosures => { force!(has_clone_closures, krate!()); }
        DepKind::HasCopyClosures => { force!(has_copy_closures, krate!()); }

        DepKind::Freevars => { force!(freevars, def_id!()); }
        DepKind::MaybeUnusedTraitImport => {
            force!(maybe_unused_trait_import, def_id!());
        }
        DepKind::MaybeUnusedExternCrates => { force!(maybe_unused_extern_crates, LOCAL_CRATE); }
        DepKind::StabilityIndex => { force!(stability_index, LOCAL_CRATE); }
        DepKind::AllCrateNums => { force!(all_crate_nums, LOCAL_CRATE); }
        DepKind::ExportedSymbols => { force!(exported_symbols, krate!()); }
        DepKind::CollectAndPartitionTranslationItems => {
            force!(collect_and_partition_translation_items, LOCAL_CRATE);
        }
        DepKind::ExportName => { force!(export_name, def_id!()); }
        DepKind::ContainsExternIndicator => {
            force!(contains_extern_indicator, def_id!());
        }
        DepKind::IsTranslatedFunction => { force!(is_translated_function, def_id!()); }
        DepKind::OutputFilenames => { force!(output_filenames, LOCAL_CRATE); }
    }

    true
}
