// Copyright 2014 The Rust Project Developers. See the COPYRIGHT
// file at the top-level directory of this distribution and at
// http://rust-lang.org/COPYRIGHT.
//
// Licensed under the Apache License, Version 2.0 <LICENSE-APACHE or
// http://www.apache.org/licenses/LICENSE-2.0> or the MIT license
// <LICENSE-MIT or http://opensource.org/licenses/MIT>, at your
// option. This file may not be copied, modified, or distributed
// except according to those terms.

//! A typesafe bitmask flag generator useful for sets of C-style bitmask flags.
//! It can be used for creating typesafe wrappers around C APIs.
//!
//! The `bitflags!` macro generates a `struct` that manages a set of flags. The
//! flags should only be defined for integer types, otherwise unexpected type
//! errors may occur at compile time.
//!
//! # Example
//!
//! ```
//! #[macro_use]
//! extern crate bitflags;
//!
//! bitflags! {
//!     struct Flags: u32 {
//!         const FLAG_A       = 0b00000001;
//!         const FLAG_B       = 0b00000010;
//!         const FLAG_C       = 0b00000100;
//!         const FLAG_ABC     = Self::FLAG_A.bits
//!                            | Self::FLAG_B.bits
//!                            | Self::FLAG_C.bits;
//!     }
//! }
//!
//! fn main() {
//!     let e1 = Flags::FLAG_A | Flags::FLAG_C;
//!     let e2 = Flags::FLAG_B | Flags::FLAG_C;
//!     assert_eq!((e1 | e2), Flags::FLAG_ABC);   // union
//!     assert_eq!((e1 & e2), Flags::FLAG_C);     // intersection
//!     assert_eq!((e1 - e2), Flags::FLAG_A);     // set difference
//!     assert_eq!(!e2, Flags::FLAG_A);           // set complement
//! }
//! ```
//!
//! See [`example_generated::Flags`](./example_generated/struct.Flags.html) for documentation of code
//! generated by the above `bitflags!` expansion.
//!
//! The generated `struct`s can also be extended with type and trait
//! implementations:
//!
//! ```
//! #[macro_use]
//! extern crate bitflags;
//!
//! use std::fmt;
//!
//! bitflags! {
//!     struct Flags: u32 {
//!         const FLAG_A   = 0b00000001;
//!         const FLAG_B   = 0b00000010;
//!     }
//! }
//!
//! impl Flags {
//!     pub fn clear(&mut self) {
//!         self.bits = 0;  // The `bits` field can be accessed from within the
//!                         // same module where the `bitflags!` macro was invoked.
//!     }
//! }
//!
//! impl fmt::Display for Flags {
//!     fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
//!         write!(f, "hi!")
//!     }
//! }
//!
//! fn main() {
//!     let mut flags = Flags::FLAG_A | Flags::FLAG_B;
//!     flags.clear();
//!     assert!(flags.is_empty());
//!     assert_eq!(format!("{}", flags), "hi!");
//!     assert_eq!(format!("{:?}", Flags::FLAG_A | Flags::FLAG_B), "FLAG_A | FLAG_B");
//!     assert_eq!(format!("{:?}", Flags::FLAG_B), "FLAG_B");
//! }
//! ```
//!
//! # Visibility
//!
//! The generated struct and its associated flag constants are not exported
//! out of the current module by default. A definition can be exported out of
//! the current module by adding `pub` before `flags`:
//!
//! ```ignore
//! #[macro_use]
//! extern crate bitflags;
//!
//! mod example {
//!     bitflags! {
//!         pub struct Flags1: u32 {
//!             const FLAG_A   = 0b00000001;
//!         }
//!     }
//!     bitflags! {
//!         struct Flags2: u32 {
//!             const FLAG_B   = 0b00000010;
//!         }
//!     }
//! }
//!
//! fn main() {
//!     let flag1 = example::Flags1::FLAG_A;
//!     let flag2 = example::Flags2::FLAG_B; // error: const `FLAG_B` is private
//! }
//! ```
//!
//! # Attributes
//!
//! Attributes can be attached to the generated `struct` by placing them
//! before the `flags` keyword.
//!
//! # Trait implementations
//!
//! The `Copy`, `Clone`, `PartialEq`, `Eq`, `PartialOrd`, `Ord` and `Hash`
//! traits automatically derived for the `struct` using the `derive` attribute.
//! Additional traits can be derived by providing an explicit `derive`
//! attribute on `flags`.
//!
//! The `Extend` and `FromIterator` traits are implemented for the `struct`,
//! too: `Extend` adds the union of the instances of the `struct` iterated over,
//! while `FromIterator` calculates the union.
//!
//! The `Binary`, `Debug`, `LowerExp`, `Octal` and `UpperExp` trait is also
//! implemented by displaying the bits value of the internal struct.
//!
//! ## Operators
//!
//! The following operator traits are implemented for the generated `struct`:
//!
//! - `BitOr` and `BitOrAssign`: union
//! - `BitAnd` and `BitAndAssign`: intersection
//! - `BitXor` and `BitXorAssign`: toggle
//! - `Sub` and `SubAssign`: set difference
//! - `Not`: set complement
//!
//! # Methods
//!
//! The following methods are defined for the generated `struct`:
//!
//! - `empty`: an empty set of flags
//! - `all`: the set of all flags
//! - `bits`: the raw value of the flags currently stored
//! - `from_bits`: convert from underlying bit representation, unless that
//!                representation contains bits that do not correspond to a flag
//! - `from_bits_truncate`: convert from underlying bit representation, dropping
//!                         any bits that do not correspond to flags
//! - `is_empty`: `true` if no flags are currently stored
//! - `is_all`: `true` if all flags are currently set
//! - `intersects`: `true` if there are flags common to both `self` and `other`
//! - `contains`: `true` all of the flags in `other` are contained within `self`
//! - `insert`: inserts the specified flags in-place
//! - `remove`: removes the specified flags in-place
//! - `toggle`: the specified flags will be inserted if not present, and removed
//!             if they are.
//! - `set`: inserts or removes the specified flags depending on the passed value
//!
//! ## Default
//!
//! The `Default` trait is not automatically implemented for the generated struct.
//!
//! If your default value is equal to `0` (which is the same value as calling `empty()`
//! on the generated struct), you can simply derive `Default`:
//!
//! ```
//! #[macro_use]
//! extern crate bitflags;
//!
//! bitflags! {
//!     // Results in default value with bits: 0
//!     #[derive(Default)]
//!     struct Flags: u32 {
//!         const FLAG_A       = 0b00000001;
//!         const FLAG_B       = 0b00000010;
//!         const FLAG_C       = 0b00000100;
//!     }
//! }
//!
//! fn main() {
//!     let derived_default: Flags = Default::default();
//!     assert_eq!(derived_default.bits(), 0);
//! }
//! ```
//!
//! If your default value is not equal to `0` you need to implement `Default` yourself:
//!
//! ```
//! #[macro_use]
//! extern crate bitflags;
//!
//! bitflags! {
//!     struct Flags: u32 {
//!         const FLAG_A       = 0b00000001;
//!         const FLAG_B       = 0b00000010;
//!         const FLAG_C       = 0b00000100;
//!     }
//! }
//!
//! // explicit `Default` implementation
//! impl Default for Flags {
//!     fn default() -> Flags {
//!         Flags::FLAG_A | Flags::FLAG_C
//!     }
//! }
//!
//! fn main() {
//!     let implemented_default: Flags = Default::default();
//!     assert_eq!(implemented_default, (Flags::FLAG_A | Flags::FLAG_C));
//! }
//! ```

#![no_std]

#![doc(html_root_url = "https://docs.rs/bitflags/1.0.0")]
// When compiled for the rustc compiler itself we want to make sure that this is
// an unstable crate.
#![cfg_attr(rustbuild, feature(staged_api))]
#![cfg_attr(rustbuild, unstable(feature = "rustc_private", issue = "27812"))]

#[cfg(test)]
#[macro_use]
extern crate std;

// Re-export libstd/libcore using an alias so that the macros can work in no_std
// crates while remaining compatible with normal crates.
#[doc(hidden)]
pub extern crate core as _core;

/// The macro used to generate the flag structure.
///
/// See the [crate level docs](../bitflags/index.html) for complete documentation.
///
/// # Example
///
/// ```
/// #[macro_use]
/// extern crate bitflags;
///
/// bitflags! {
///     struct Flags: u32 {
///         const FLAG_A       = 0b00000001;
///         const FLAG_B       = 0b00000010;
///         const FLAG_C       = 0b00000100;
///         const FLAG_ABC     = Self::FLAG_A.bits
///                            | Self::FLAG_B.bits
///                            | Self::FLAG_C.bits;
///     }
/// }
///
/// fn main() {
///     let e1 = Flags::FLAG_A | Flags::FLAG_C;
///     let e2 = Flags::FLAG_B | Flags::FLAG_C;
///     assert_eq!((e1 | e2), Flags::FLAG_ABC);   // union
///     assert_eq!((e1 & e2), Flags::FLAG_C);     // intersection
///     assert_eq!((e1 - e2), Flags::FLAG_A);     // set difference
///     assert_eq!(!e2, Flags::FLAG_A);           // set complement
/// }
/// ```
///
/// The generated `struct`s can also be extended with type and trait
/// implementations:
///
/// ```
/// #[macro_use]
/// extern crate bitflags;
///
/// use std::fmt;
///
/// bitflags! {
///     struct Flags: u32 {
///         const FLAG_A   = 0b00000001;
///         const FLAG_B   = 0b00000010;
///     }
/// }
///
/// impl Flags {
///     pub fn clear(&mut self) {
///         self.bits = 0;  // The `bits` field can be accessed from within the
///                         // same module where the `bitflags!` macro was invoked.
///     }
/// }
///
/// impl fmt::Display for Flags {
///     fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
///         write!(f, "hi!")
///     }
/// }
///
/// fn main() {
///     let mut flags = Flags::FLAG_A | Flags::FLAG_B;
///     flags.clear();
///     assert!(flags.is_empty());
///     assert_eq!(format!("{}", flags), "hi!");
///     assert_eq!(format!("{:?}", Flags::FLAG_A | Flags::FLAG_B), "FLAG_A | FLAG_B");
///     assert_eq!(format!("{:?}", Flags::FLAG_B), "FLAG_B");
/// }
/// ```
#[macro_export]
macro_rules! bitflags {
    (
        $(#[$outer:meta])*
        pub struct $BitFlags:ident: $T:ty {
            $(
                $(#[$inner:ident $($args:tt)*])*
                const $Flag:ident = $value:expr;
            )+
        }
    ) => {
        #[derive(Copy, PartialEq, Eq, Clone, PartialOrd, Ord, Hash)]
        $(#[$outer])*
        pub struct $BitFlags {
            bits: $T,
        }

        __impl_bitflags! {
            struct $BitFlags: $T {
                $(
                    $(#[$inner $($args)*])*
                    const $Flag = $value;
                )+
            }
        }
    };
    (
        $(#[$outer:meta])*
        struct $BitFlags:ident: $T:ty {
            $(
                $(#[$inner:ident $($args:tt)*])*
                const $Flag:ident = $value:expr;
            )+
        }
    ) => {
        #[derive(Copy, PartialEq, Eq, Clone, PartialOrd, Ord, Hash)]
        $(#[$outer])*
        struct $BitFlags {
            bits: $T,
        }

        __impl_bitflags! {
            struct $BitFlags: $T {
                $(
                    $(#[$inner $($args)*])*
                    const $Flag = $value;
                )+
            }
        }
    };
}

#[macro_export]
#[doc(hidden)]
macro_rules! __impl_bitflags {
    (
        struct $BitFlags:ident: $T:ty {
            $(
                $(#[$attr:ident $($args:tt)*])*
                const $Flag:ident = $value:expr;
            )+
        }
    ) => {
        impl $crate::_core::fmt::Debug for $BitFlags {
            fn fmt(&self, f: &mut $crate::_core::fmt::Formatter) -> $crate::_core::fmt::Result {
                // This convoluted approach is to handle #[cfg]-based flag
                // omission correctly. For example it needs to support:
                //
                //    #[cfg(unix)] const A: Flag = /* ... */;
                //    #[cfg(windows)] const B: Flag = /* ... */;

                // Unconditionally define a check for every flag, even disabled
                // ones.
                #[allow(non_snake_case)]
                trait __BitFlags {
                    $(
                        fn $Flag(&self) -> bool { false }
                    )+
                }

                // Conditionally override the check for just those flags that
                // are not #[cfg]ed away.
                impl __BitFlags for $BitFlags {
                    $(
                        __impl_bitflags! {
                            #[allow(deprecated)]
                            $(? #[$attr $($args)*])*
                            fn $Flag(&self) -> bool {
                                self.bits & Self::$Flag.bits == Self::$Flag.bits
                            }
                        }
                    )+
                }

                let mut first = true;
                $(
                    if <$BitFlags as __BitFlags>::$Flag(self) {
                        if !first {
                            try!(f.write_str(" | "));
                        }
                        first = false;
                        try!(f.write_str(stringify!($Flag)));
                    }
                )+
                if first {
                    try!(f.write_str("(empty)"));
                }
                Ok(())
            }
        }
        impl $crate::_core::fmt::Binary for $BitFlags {
            fn fmt(&self, f: &mut $crate::_core::fmt::Formatter) -> $crate::_core::fmt::Result {
                $crate::_core::fmt::Binary::fmt(&self.bits, f)
            }
        }
        impl $crate::_core::fmt::Octal for $BitFlags {
            fn fmt(&self, f: &mut $crate::_core::fmt::Formatter) -> $crate::_core::fmt::Result {
                $crate::_core::fmt::Octal::fmt(&self.bits, f)
            }
        }
        impl $crate::_core::fmt::LowerHex for $BitFlags {
            fn fmt(&self, f: &mut $crate::_core::fmt::Formatter) -> $crate::_core::fmt::Result {
                $crate::_core::fmt::LowerHex::fmt(&self.bits, f)
            }
        }
        impl $crate::_core::fmt::UpperHex for $BitFlags {
            fn fmt(&self, f: &mut $crate::_core::fmt::Formatter) -> $crate::_core::fmt::Result {
                $crate::_core::fmt::UpperHex::fmt(&self.bits, f)
            }
        }

        #[allow(dead_code)]
        impl $BitFlags {
            $(
                $(#[$attr $($args)*])*
                pub const $Flag: $BitFlags = $BitFlags { bits: $value };
            )+

            /// Returns an empty set of flags.
            #[inline]
            pub fn empty() -> $BitFlags {
                $BitFlags { bits: 0 }
            }

            /// Returns the set containing all flags.
            #[inline]
            pub fn all() -> $BitFlags {
                // See `Debug::fmt` for why this approach is taken.
                #[allow(non_snake_case)]
                trait __BitFlags {
                    $(
                        fn $Flag() -> $T { 0 }
                    )+
                }
                impl __BitFlags for $BitFlags {
                    $(
                        __impl_bitflags! {
                            #[allow(deprecated)]
                            $(? #[$attr $($args)*])*
                            fn $Flag() -> $T { Self::$Flag.bits }
                        }
                    )+
                }
                $BitFlags { bits: $(<$BitFlags as __BitFlags>::$Flag())|+ }
            }

            /// Returns the raw value of the flags currently stored.
            #[inline]
            pub fn bits(&self) -> $T {
                self.bits
            }

            /// Convert from underlying bit representation, unless that
            /// representation contains bits that do not correspond to a flag.
            #[inline]
            pub fn from_bits(bits: $T) -> $crate::_core::option::Option<$BitFlags> {
                if (bits & !$BitFlags::all().bits()) == 0 {
                    $crate::_core::option::Option::Some($BitFlags { bits: bits })
                } else {
                    $crate::_core::option::Option::None
                }
            }

            /// Convert from underlying bit representation, dropping any bits
            /// that do not correspond to flags.
            #[inline]
            pub fn from_bits_truncate(bits: $T) -> $BitFlags {
                $BitFlags { bits: bits } & $BitFlags::all()
            }

            /// Returns `true` if no flags are currently stored.
            #[inline]
            pub fn is_empty(&self) -> bool {
                *self == $BitFlags::empty()
            }

            /// Returns `true` if all flags are currently set.
            #[inline]
            pub fn is_all(&self) -> bool {
                *self == $BitFlags::all()
            }

            /// Returns `true` if there are flags common to both `self` and `other`.
            #[inline]
            pub fn intersects(&self, other: $BitFlags) -> bool {
                !(*self & other).is_empty()
            }

            /// Returns `true` all of the flags in `other` are contained within `self`.
            #[inline]
            pub fn contains(&self, other: $BitFlags) -> bool {
                (*self & other) == other
            }

            /// Inserts the specified flags in-place.
            #[inline]
            pub fn insert(&mut self, other: $BitFlags) {
                self.bits |= other.bits;
            }

            /// Removes the specified flags in-place.
            #[inline]
            pub fn remove(&mut self, other: $BitFlags) {
                self.bits &= !other.bits;
            }

            /// Toggles the specified flags in-place.
            #[inline]
            pub fn toggle(&mut self, other: $BitFlags) {
                self.bits ^= other.bits;
            }

            /// Inserts or removes the specified flags depending on the passed value.
            #[inline]
            pub fn set(&mut self, other: $BitFlags, value: bool) {
                if value {
                    self.insert(other);
                } else {
                    self.remove(other);
                }
            }
        }

        impl $crate::_core::ops::BitOr for $BitFlags {
            type Output = $BitFlags;

            /// Returns the union of the two sets of flags.
            #[inline]
            fn bitor(self, other: $BitFlags) -> $BitFlags {
                $BitFlags { bits: self.bits | other.bits }
            }
        }

        impl $crate::_core::ops::BitOrAssign for $BitFlags {

            /// Adds the set of flags.
            #[inline]
            fn bitor_assign(&mut self, other: $BitFlags) {
                self.bits |= other.bits;
            }
        }

        impl $crate::_core::ops::BitXor for $BitFlags {
            type Output = $BitFlags;

            /// Returns the left flags, but with all the right flags toggled.
            #[inline]
            fn bitxor(self, other: $BitFlags) -> $BitFlags {
                $BitFlags { bits: self.bits ^ other.bits }
            }
        }

        impl $crate::_core::ops::BitXorAssign for $BitFlags {

            /// Toggles the set of flags.
            #[inline]
            fn bitxor_assign(&mut self, other: $BitFlags) {
                self.bits ^= other.bits;
            }
        }

        impl $crate::_core::ops::BitAnd for $BitFlags {
            type Output = $BitFlags;

            /// Returns the intersection between the two sets of flags.
            #[inline]
            fn bitand(self, other: $BitFlags) -> $BitFlags {
                $BitFlags { bits: self.bits & other.bits }
            }
        }

        impl $crate::_core::ops::BitAndAssign for $BitFlags {

            /// Disables all flags disabled in the set.
            #[inline]
            fn bitand_assign(&mut self, other: $BitFlags) {
                self.bits &= other.bits;
            }
        }

        impl $crate::_core::ops::Sub for $BitFlags {
            type Output = $BitFlags;

            /// Returns the set difference of the two sets of flags.
            #[inline]
            fn sub(self, other: $BitFlags) -> $BitFlags {
                $BitFlags { bits: self.bits & !other.bits }
            }
        }

        impl $crate::_core::ops::SubAssign for $BitFlags {

            /// Disables all flags enabled in the set.
            #[inline]
            fn sub_assign(&mut self, other: $BitFlags) {
                self.bits &= !other.bits;
            }
        }

        impl $crate::_core::ops::Not for $BitFlags {
            type Output = $BitFlags;

            /// Returns the complement of this set of flags.
            #[inline]
            fn not(self) -> $BitFlags {
                $BitFlags { bits: !self.bits } & $BitFlags::all()
            }
        }

        impl $crate::_core::iter::Extend<$BitFlags> for $BitFlags {
            fn extend<T: $crate::_core::iter::IntoIterator<Item=$BitFlags>>(&mut self, iterator: T) {
                for item in iterator {
                    self.insert(item)
                }
            }
        }

        impl $crate::_core::iter::FromIterator<$BitFlags> for $BitFlags {
            fn from_iter<T: $crate::_core::iter::IntoIterator<Item=$BitFlags>>(iterator: T) -> $BitFlags {
                let mut result = Self::empty();
                result.extend(iterator);
                result
            }
        }
    };

    // Every attribute that the user writes on a const is applied to the
    // corresponding const that we generate, but within the implementation of
    // Debug and all() we want to ignore everything but #[cfg] attributes. In
    // particular, including a #[deprecated] attribute on those items would fail
    // to compile.
    // https://github.com/rust-lang-nursery/bitflags/issues/109
    //
    // Input:
    //
    //     ? #[cfg(feature = "advanced")]
    //     ? #[deprecated(note = "Use somthing else.")]
    //     ? #[doc = r"High quality documentation."]
    //     fn f() -> i32 { /* ... */ }
    //
    // Output:
    //
    //     #[cfg(feature = "advanced")]
    //     fn f() -> i32 { /* ... */ }
    (
        $(#[$filtered:meta])*
        ? #[cfg $($cfgargs:tt)*]
        $(? #[$rest:ident $($restargs:tt)*])*
        fn $($item:tt)*
    ) => {
        __impl_bitflags! {
            $(#[$filtered])*
            #[cfg $($cfgargs)*]
            $(? #[$rest $($restargs)*])*
            fn $($item)*
        }
    };
    (
        $(#[$filtered:meta])*
        // $next != `cfg`
        ? #[$next:ident $($nextargs:tt)*]
        $(? #[$rest:ident $($restargs:tt)*])*
        fn $($item:tt)*
    ) => {
        __impl_bitflags! {
            $(#[$filtered])*
            // $next filtered out
            $(? #[$rest $($restargs)*])*
            fn $($item)*
        }
    };
    (
        $(#[$filtered:meta])*
        fn $($item:tt)*
    ) => {
        $(#[$filtered])*
        fn $($item)*
    };
}

#[cfg(feature = "example_generated")]
pub mod example_generated;

#[cfg(test)]
mod tests {
    use std::hash::{Hash, Hasher};
    use std::collections::hash_map::DefaultHasher;

    bitflags! {
        #[doc = "> The first principle is that you must not fool yourself — and"]
        #[doc = "> you are the easiest person to fool."]
        #[doc = "> "]
        #[doc = "> - Richard Feynman"]
        struct Flags: u32 {
            const FLAG_A       = 0b00000001;
            #[doc = "<pcwalton> macros are way better at generating code than trans is"]
            const FLAG_B       = 0b00000010;
            const FLAG_C       = 0b00000100;
            #[doc = "* cmr bed"]
            #[doc = "* strcat table"]
            #[doc = "<strcat> wait what?"]
            const FLAG_ABC     = Self::FLAG_A.bits
                               | Self::FLAG_B.bits
                               | Self::FLAG_C.bits;
        }
    }

    bitflags! {
        struct _CfgFlags: u32 {
            #[cfg(windows)]
            const _CFG_A = 0b01;
            #[cfg(unix)]
            const _CFG_B = 0b01;
            #[cfg(windows)]
            const _CFG_C = _CFG_A.bits | 0b10;
        }
    }

    bitflags! {
        struct AnotherSetOfFlags: i8 {
            const ANOTHER_FLAG = -1_i8;
        }
    }

    bitflags! {
        struct LongFlags: u32 {
            const LONG_FLAG_A = 0b1111111111111111;
        }
    }

    #[test]
    fn test_bits(){
        assert_eq!(Flags::empty().bits(), 0b00000000);
        assert_eq!(Flags::FLAG_A.bits(), 0b00000001);
        assert_eq!(Flags::FLAG_ABC.bits(), 0b00000111);

        assert_eq!(AnotherSetOfFlags::empty().bits(), 0b00);
        assert_eq!(AnotherSetOfFlags::ANOTHER_FLAG.bits(), !0_i8);
    }

    #[test]
    fn test_from_bits() {
        assert_eq!(Flags::from_bits(0), Some(Flags::empty()));
        assert_eq!(Flags::from_bits(0b1), Some(Flags::FLAG_A));
        assert_eq!(Flags::from_bits(0b10), Some(Flags::FLAG_B));
        assert_eq!(Flags::from_bits(0b11), Some(Flags::FLAG_A | Flags::FLAG_B));
        assert_eq!(Flags::from_bits(0b1000), None);

        assert_eq!(AnotherSetOfFlags::from_bits(!0_i8), Some(AnotherSetOfFlags::ANOTHER_FLAG));
    }

    #[test]
    fn test_from_bits_truncate() {
        assert_eq!(Flags::from_bits_truncate(0), Flags::empty());
        assert_eq!(Flags::from_bits_truncate(0b1), Flags::FLAG_A);
        assert_eq!(Flags::from_bits_truncate(0b10), Flags::FLAG_B);
        assert_eq!(Flags::from_bits_truncate(0b11), (Flags::FLAG_A | Flags::FLAG_B));
        assert_eq!(Flags::from_bits_truncate(0b1000), Flags::empty());
        assert_eq!(Flags::from_bits_truncate(0b1001), Flags::FLAG_A);

        assert_eq!(AnotherSetOfFlags::from_bits_truncate(0_i8), AnotherSetOfFlags::empty());
    }

    #[test]
    fn test_is_empty(){
        assert!(Flags::empty().is_empty());
        assert!(!Flags::FLAG_A.is_empty());
        assert!(!Flags::FLAG_ABC.is_empty());

        assert!(!AnotherSetOfFlags::ANOTHER_FLAG.is_empty());
    }

    #[test]
    fn test_is_all() {
        assert!(Flags::all().is_all());
        assert!(!Flags::FLAG_A.is_all());
        assert!(Flags::FLAG_ABC.is_all());

        assert!(AnotherSetOfFlags::ANOTHER_FLAG.is_all());
    }

    #[test]
    fn test_two_empties_do_not_intersect() {
        let e1 = Flags::empty();
        let e2 = Flags::empty();
        assert!(!e1.intersects(e2));

        assert!(AnotherSetOfFlags::ANOTHER_FLAG.intersects(AnotherSetOfFlags::ANOTHER_FLAG));
    }

    #[test]
    fn test_empty_does_not_intersect_with_full() {
        let e1 = Flags::empty();
        let e2 = Flags::FLAG_ABC;
        assert!(!e1.intersects(e2));
    }

    #[test]
    fn test_disjoint_intersects() {
        let e1 = Flags::FLAG_A;
        let e2 = Flags::FLAG_B;
        assert!(!e1.intersects(e2));
    }

    #[test]
    fn test_overlapping_intersects() {
        let e1 = Flags::FLAG_A;
        let e2 = Flags::FLAG_A | Flags::FLAG_B;
        assert!(e1.intersects(e2));
    }

    #[test]
    fn test_contains() {
        let e1 = Flags::FLAG_A;
        let e2 = Flags::FLAG_A | Flags::FLAG_B;
        assert!(!e1.contains(e2));
        assert!(e2.contains(e1));
        assert!(Flags::FLAG_ABC.contains(e2));

        assert!(AnotherSetOfFlags::ANOTHER_FLAG.contains(AnotherSetOfFlags::ANOTHER_FLAG));
    }

    #[test]
    fn test_insert(){
        let mut e1 = Flags::FLAG_A;
        let e2 = Flags::FLAG_A | Flags::FLAG_B;
        e1.insert(e2);
        assert_eq!(e1, e2);

        let mut e3 = AnotherSetOfFlags::empty();
        e3.insert(AnotherSetOfFlags::ANOTHER_FLAG);
        assert_eq!(e3, AnotherSetOfFlags::ANOTHER_FLAG);
    }

    #[test]
    fn test_remove(){
        let mut e1 = Flags::FLAG_A | Flags::FLAG_B;
        let e2 = Flags::FLAG_A | Flags::FLAG_C;
        e1.remove(e2);
        assert_eq!(e1, Flags::FLAG_B);

        let mut e3 = AnotherSetOfFlags::ANOTHER_FLAG;
        e3.remove(AnotherSetOfFlags::ANOTHER_FLAG);
        assert_eq!(e3, AnotherSetOfFlags::empty());
    }

    #[test]
    fn test_operators() {
        let e1 = Flags::FLAG_A | Flags::FLAG_C;
        let e2 = Flags::FLAG_B | Flags::FLAG_C;
        assert_eq!((e1 | e2), Flags::FLAG_ABC);     // union
        assert_eq!((e1 & e2), Flags::FLAG_C);       // intersection
        assert_eq!((e1 - e2), Flags::FLAG_A);       // set difference
        assert_eq!(!e2, Flags::FLAG_A);             // set complement
        assert_eq!(e1 ^ e2, Flags::FLAG_A | Flags::FLAG_B); // toggle
        let mut e3 = e1;
        e3.toggle(e2);
        assert_eq!(e3, Flags::FLAG_A | Flags::FLAG_B);

        let mut m4 = AnotherSetOfFlags::empty();
        m4.toggle(AnotherSetOfFlags::empty());
        assert_eq!(m4, AnotherSetOfFlags::empty());
    }

    #[test]
    fn test_set() {
        let mut e1 = Flags::FLAG_A | Flags::FLAG_C;
        e1.set(Flags::FLAG_B, true);
        e1.set(Flags::FLAG_C, false);

        assert_eq!(e1, Flags::FLAG_A | Flags::FLAG_B);
    }

    #[test]
    fn test_assignment_operators() {
        let mut m1 = Flags::empty();
        let e1 = Flags::FLAG_A | Flags::FLAG_C;
        // union
        m1 |= Flags::FLAG_A;
        assert_eq!(m1, Flags::FLAG_A);
        // intersection
        m1 &= e1;
        assert_eq!(m1, Flags::FLAG_A);
        // set difference
        m1 -= m1;
        assert_eq!(m1, Flags::empty());
        // toggle
        m1 ^= e1;
        assert_eq!(m1, e1);
    }

    #[test]
    fn test_extend() {
        let mut flags;

        flags = Flags::empty();
        flags.extend([].iter().cloned());
        assert_eq!(flags, Flags::empty());

        flags = Flags::empty();
        flags.extend([Flags::FLAG_A, Flags::FLAG_B].iter().cloned());
        assert_eq!(flags, Flags::FLAG_A | Flags::FLAG_B);

        flags = Flags::FLAG_A;
        flags.extend([Flags::FLAG_A, Flags::FLAG_B].iter().cloned());
        assert_eq!(flags, Flags::FLAG_A | Flags::FLAG_B);

        flags = Flags::FLAG_B;
        flags.extend([Flags::FLAG_A, Flags::FLAG_ABC].iter().cloned());
        assert_eq!(flags, Flags::FLAG_ABC);
    }

    #[test]
    fn test_from_iterator() {
        assert_eq!([].iter().cloned().collect::<Flags>(), Flags::empty());
        assert_eq!([Flags::FLAG_A, Flags::FLAG_B].iter().cloned().collect::<Flags>(),
                   Flags::FLAG_A | Flags::FLAG_B);
        assert_eq!([Flags::FLAG_A, Flags::FLAG_ABC].iter().cloned().collect::<Flags>(),
                   Flags::FLAG_ABC);
    }

    #[test]
    fn test_lt() {
        let mut a = Flags::empty();
        let mut b = Flags::empty();

        assert!(!(a < b) && !(b < a));
        b = Flags::FLAG_B;
        assert!(a < b);
        a = Flags::FLAG_C;
        assert!(!(a < b) && b < a);
        b = Flags::FLAG_C | Flags::FLAG_B;
        assert!(a < b);
    }

    #[test]
    fn test_ord() {
        let mut a = Flags::empty();
        let mut b = Flags::empty();

        assert!(a <= b && a >= b);
        a = Flags::FLAG_A;
        assert!(a > b && a >= b);
        assert!(b < a && b <= a);
        b = Flags::FLAG_B;
        assert!(b > a && b >= a);
        assert!(a < b && a <= b);
    }

    fn hash<T: Hash>(t: &T) -> u64 {
        let mut s = DefaultHasher::new();
        t.hash(&mut s);
        s.finish()
    }

    #[test]
    fn test_hash() {
        let mut x = Flags::empty();
        let mut y = Flags::empty();
        assert_eq!(hash(&x), hash(&y));
        x = Flags::all();
        y = Flags::FLAG_ABC;
        assert_eq!(hash(&x), hash(&y));
    }

    #[test]
    fn test_debug() {
        assert_eq!(format!("{:?}", Flags::FLAG_A | Flags::FLAG_B), "FLAG_A | FLAG_B");
        assert_eq!(format!("{:?}", Flags::empty()), "(empty)");
        assert_eq!(format!("{:?}", Flags::FLAG_ABC), "FLAG_A | FLAG_B | FLAG_C | FLAG_ABC");
    }

    #[test]
    fn test_binary() {
        assert_eq!(format!("{:b}", Flags::FLAG_ABC), "111");
        assert_eq!(format!("{:#b}", Flags::FLAG_ABC), "0b111");
    }

    #[test]
    fn test_octal() {
        assert_eq!(format!("{:o}", LongFlags::LONG_FLAG_A), "177777");
        assert_eq!(format!("{:#o}", LongFlags::LONG_FLAG_A), "0o177777");
    }

    #[test]
    fn test_lowerhex() {
        assert_eq!(format!("{:x}", LongFlags::LONG_FLAG_A), "ffff");
        assert_eq!(format!("{:#x}", LongFlags::LONG_FLAG_A), "0xffff");
    }

    #[test]
    fn test_upperhex() {
        assert_eq!(format!("{:X}", LongFlags::LONG_FLAG_A), "FFFF");
        assert_eq!(format!("{:#X}", LongFlags::LONG_FLAG_A), "0xFFFF");
    }

    mod submodule {
        bitflags! {
            pub struct PublicFlags: i8 {
                const FLAG_X = 0;
            }
        }
        bitflags! {
            struct PrivateFlags: i8 {
                const FLAG_Y = 0;
            }
        }

        #[test]
        fn test_private() {

            let _ = PrivateFlags::FLAG_Y;
        }
    }

    #[test]
    fn test_public() {
        let _ = submodule::PublicFlags::FLAG_X;
    }

    mod t1 {
        mod foo {
            pub type Bar = i32;
        }

        bitflags! {
            /// baz
            struct Flags: foo::Bar {
                const A       = 0b00000001;
                #[cfg(foo)]
                const B       = 0b00000010;
                #[cfg(foo)]
                const C       = 0b00000010;
            }
        }
    }

    #[test]
    fn test_in_function() {
        bitflags! {
           struct Flags: u8 {
                const A = 1;
                #[cfg(any())] // false
                const B = 2;
            }
        }
        assert_eq!(Flags::all(), Flags::A);
        assert_eq!(format!("{:?}", Flags::A), "A");
    }

    #[test]
    fn test_deprecated() {
        bitflags! {
            pub struct TestFlags: u32 {
                #[deprecated(note = "Use something else.")]
                const FLAG_ONE = 1;
            }
        }
    }
}
