use support::git;
use support::paths;
use support::registry::Package;
use support::{basic_manifest, execs, project};
use support::hamcrest::assert_that;

#[test]
fn override_simple() {
    Package::new("bar", "0.1.0").publish();

    let bar = git::repo(&paths::root().join("override"))
        .file("Cargo.toml", &basic_manifest("bar", "0.1.0"))
        .file("src/lib.rs", "pub fn bar() {}")
        .build();

    let p = project()
        .file(
            "Cargo.toml",
            &format!(
                r#"
            [package]
            name = "foo"
            version = "0.0.1"
            authors = []

            [dependencies]
            bar = "0.1.0"

            [replace]
            "bar:0.1.0" = {{ git = '{}' }}
        "#,
                bar.url()
            ),
        )
        .file("src/lib.rs", "extern crate bar; pub fn foo() { bar::bar(); }")
        .build();

    assert_that(
        p.cargo("build"),
        execs().with_stderr(
            "\
[UPDATING] registry `file://[..]`
[UPDATING] git repository `[..]`
[COMPILING] bar v0.1.0 (file://[..])
[COMPILING] foo v0.0.1 (file://[..])
[FINISHED] dev [unoptimized + debuginfo] target(s) in [..]
",
        ),
    );
}

#[test]
fn missing_version() {
    let p = project()
        .file(
            "Cargo.toml",
            r#"
            [package]
            name = "foo"
            version = "0.0.1"
            authors = []

            [dependencies]
            bar = "0.1.0"

            [replace]
            bar = { git = 'https://example.com' }
        "#,
        )
        .file("src/lib.rs", "")
        .build();

    assert_that(
        p.cargo("build"),
        execs().with_status(101).with_stderr(
            "\
error: failed to parse manifest at `[..]`

Caused by:
  replacements must specify a version to replace, but `[..]bar` does not
",
        ),
    );
}

#[test]
fn invalid_semver_version() {
    let p = project()
        .file(
            "Cargo.toml",
            r#"
            [package]
            name = "foo"
            version = "0.0.1"
            authors = []

            [dependencies]
            bar = "*"

            [replace]
            "bar:*" = { git = 'https://example.com' }
        "#,
        )
        .file("src/lib.rs", "")
        .build();

    assert_that(
        p.cargo("build"),
        execs().with_status(101).with_stderr_contains(
            "\
error: failed to parse manifest at `[..]`

Caused by:
  replacements must specify a valid semver version to replace, but `bar:*` does not
",
        ),
    );
}

#[test]
fn different_version() {
    Package::new("bar", "0.2.0").publish();
    Package::new("bar", "0.1.0").publish();

    let p = project()
        .file(
            "Cargo.toml",
            r#"
            [package]
            name = "foo"
            version = "0.0.1"
            authors = []

            [dependencies]
            bar = "0.1.0"

            [replace]
            "bar:0.1.0" = "0.2.0"
        "#,
        )
        .file("src/lib.rs", "")
        .build();

    assert_that(
        p.cargo("build"),
        execs().with_status(101).with_stderr(
            "\
error: failed to parse manifest at `[..]`

Caused by:
  replacements cannot specify a version requirement, but found one for [..]
",
        ),
    );
}

#[test]
fn transitive() {
    Package::new("bar", "0.1.0").publish();
    Package::new("baz", "0.2.0")
        .dep("bar", "0.1.0")
        .file("src/lib.rs", "extern crate bar; fn baz() { bar::bar(); }")
        .publish();

    let foo = git::repo(&paths::root().join("override"))
        .file("Cargo.toml", &basic_manifest("bar", "0.1.0"))
        .file("src/lib.rs", "pub fn bar() {}")
        .build();

    let p = project()
        .file(
            "Cargo.toml",
            &format!(
                r#"
            [package]
            name = "foo"
            version = "0.0.1"
            authors = []

            [dependencies]
            baz = "0.2.0"

            [replace]
            "bar:0.1.0" = {{ git = '{}' }}
        "#,
                foo.url()
            ),
        )
        .file("src/lib.rs", "")
        .build();

    assert_that(
        p.cargo("build"),
        execs().with_stderr(
            "\
[UPDATING] registry `file://[..]`
[UPDATING] git repository `[..]`
[DOWNLOADING] baz v0.2.0 (registry [..])
[COMPILING] bar v0.1.0 (file://[..])
[COMPILING] baz v0.2.0
[COMPILING] foo v0.0.1 (file://[..])
[FINISHED] dev [unoptimized + debuginfo] target(s) in [..]
",
        ),
    );

    assert_that(p.cargo("build"), execs().with_stdout(""));
}

#[test]
fn persists_across_rebuilds() {
    Package::new("bar", "0.1.0").publish();

    let foo = git::repo(&paths::root().join("override"))
        .file("Cargo.toml", &basic_manifest("bar", "0.1.0"))
        .file("src/lib.rs", "pub fn bar() {}")
        .build();

    let p = project()
        .file(
            "Cargo.toml",
            &format!(
                r#"
            [package]
            name = "foo"
            version = "0.0.1"
            authors = []

            [dependencies]
            bar = "0.1.0"

            [replace]
            "bar:0.1.0" = {{ git = '{}' }}
        "#,
                foo.url()
            ),
        )
        .file("src/lib.rs", "extern crate bar; pub fn foo() { bar::bar(); }")
        .build();

    assert_that(
        p.cargo("build"),
        execs().with_stderr(
            "\
[UPDATING] registry `file://[..]`
[UPDATING] git repository `file://[..]`
[COMPILING] bar v0.1.0 (file://[..])
[COMPILING] foo v0.0.1 (file://[..])
[FINISHED] dev [unoptimized + debuginfo] target(s) in [..]
",
        ),
    );

    assert_that(p.cargo("build"), execs().with_stdout(""));
}

#[test]
fn replace_registry_with_path() {
    Package::new("bar", "0.1.0").publish();

    let _ = project().at("bar")
        .file("Cargo.toml", &basic_manifest("bar", "0.1.0"))
        .file("src/lib.rs", "pub fn bar() {}")
        .build();

    let p = project()
        .file(
            "Cargo.toml",
            r#"
            [package]
            name = "foo"
            version = "0.0.1"
            authors = []

            [dependencies]
            bar = "0.1.0"

            [replace]
            "bar:0.1.0" = { path = "../bar" }
        "#,
        )
        .file("src/lib.rs", "extern crate bar; pub fn foo() { bar::bar(); }")
        .build();

    assert_that(
        p.cargo("build"),
        execs().with_stderr(
            "\
[UPDATING] registry `file://[..]`
[COMPILING] bar v0.1.0 (file://[..])
[COMPILING] foo v0.0.1 (file://[..])
[FINISHED] dev [unoptimized + debuginfo] target(s) in [..]
",
        ),
    );
}

#[test]
fn use_a_spec_to_select() {
    Package::new("baz", "0.1.1")
        .file("src/lib.rs", "pub fn baz1() {}")
        .publish();
    Package::new("baz", "0.2.0").publish();
    Package::new("bar", "0.1.1")
        .dep("baz", "0.2")
        .file("src/lib.rs", "extern crate baz; pub fn bar() { baz::baz3(); }")
        .publish();

    let foo = git::repo(&paths::root().join("override"))
        .file("Cargo.toml", &basic_manifest("baz", "0.2.0"))
        .file("src/lib.rs", "pub fn baz3() {}")
        .build();

    let p = project()
        .file(
            "Cargo.toml",
            &format!(
                r#"
            [package]
            name = "foo"
            version = "0.0.1"
            authors = []

            [dependencies]
            bar = "0.1"
            baz = "0.1"

            [replace]
            "baz:0.2.0" = {{ git = '{}' }}
        "#,
                foo.url()
            ),
        )
        .file(
            "src/lib.rs",
            "
            extern crate bar;
            extern crate baz;

            pub fn local() {
                baz::baz1();
                bar::bar();
            }
        ",
        )
        .build();

    assert_that(
        p.cargo("build"),
        execs().with_stderr(
            "\
[UPDATING] registry `file://[..]`
[UPDATING] git repository `[..]`
[DOWNLOADING] [..]
[DOWNLOADING] [..]
[COMPILING] [..]
[COMPILING] [..]
[COMPILING] [..]
[COMPILING] foo v0.0.1 (file://[..])
[FINISHED] dev [unoptimized + debuginfo] target(s) in [..]
",
        ),
    );
}

#[test]
fn override_adds_some_deps() {
    Package::new("baz", "0.1.1").publish();
    Package::new("bar", "0.1.0").publish();

    let foo = git::repo(&paths::root().join("override"))
        .file(
            "Cargo.toml",
            r#"
            [package]
            name = "bar"
            version = "0.1.0"
            authors = []

            [dependencies]
            baz = "0.1"
        "#,
        )
        .file("src/lib.rs", "")
        .build();

    let p = project()
        .file(
            "Cargo.toml",
            &format!(
                r#"
            [package]
            name = "foo"
            version = "0.0.1"
            authors = []

            [dependencies]
            bar = "0.1"

            [replace]
            "bar:0.1.0" = {{ git = '{}' }}
        "#,
                foo.url()
            ),
        )
        .file("src/lib.rs", "")
        .build();

    assert_that(
        p.cargo("build"),
        execs().with_stderr(
            "\
[UPDATING] registry `file://[..]`
[UPDATING] git repository `[..]`
[DOWNLOADING] baz v0.1.1 (registry [..])
[COMPILING] baz v0.1.1
[COMPILING] bar v0.1.0 ([..])
[COMPILING] foo v0.0.1 (file://[..])
[FINISHED] dev [unoptimized + debuginfo] target(s) in [..]
",
        ),
    );

    assert_that(p.cargo("build"), execs().with_stdout(""));

    Package::new("baz", "0.1.2").publish();
    assert_that(
        p.cargo("update")
            .arg("-p")
            .arg(&format!("{}#bar", foo.url())),
        execs().with_stderr(
            "\
[UPDATING] git repository `file://[..]`
[UPDATING] registry `file://[..]`
",
        ),
    );
    assert_that(
        p.cargo("update")
            .arg("-p")
            .arg("https://github.com/rust-lang/crates.io-index#bar"),
        execs().with_stderr(
            "\
[UPDATING] registry `file://[..]`
",
        ),
    );

    assert_that(p.cargo("build"), execs().with_stdout(""));
}

#[test]
fn locked_means_locked_yes_no_seriously_i_mean_locked() {
    // this in theory exercises #2041
    Package::new("baz", "0.1.0").publish();
    Package::new("baz", "0.2.0").publish();
    Package::new("bar", "0.1.0").publish();

    let foo = git::repo(&paths::root().join("override"))
        .file(
            "Cargo.toml",
            r#"
            [package]
            name = "bar"
            version = "0.1.0"
            authors = []

            [dependencies]
            baz = "*"
        "#,
        )
        .file("src/lib.rs", "")
        .build();

    let p = project()
        .file(
            "Cargo.toml",
            &format!(
                r#"
            [package]
            name = "foo"
            version = "0.0.1"
            authors = []

            [dependencies]
            bar = "0.1"
            baz = "0.1"

            [replace]
            "bar:0.1.0" = {{ git = '{}' }}
        "#,
                foo.url()
            ),
        )
        .file("src/lib.rs", "")
        .build();

    assert_that(p.cargo("build"), execs());

    assert_that(p.cargo("build"), execs().with_stdout(""));
    assert_that(p.cargo("build"), execs().with_stdout(""));
}

#[test]
fn override_wrong_name() {
    Package::new("baz", "0.1.0").publish();

    let foo = git::repo(&paths::root().join("override"))
        .file("Cargo.toml", &basic_manifest("bar", "0.1.0"))
        .file("src/lib.rs", "")
        .build();

    let p = project()
        .file(
            "Cargo.toml",
            &format!(
                r#"
            [package]
            name = "foo"
            version = "0.0.1"
            authors = []

            [dependencies]
            baz = "0.1"

            [replace]
            "baz:0.1.0" = {{ git = '{}' }}
        "#,
                foo.url()
            ),
        )
        .file("src/lib.rs", "")
        .build();

    assert_that(
        p.cargo("build"),
        execs().with_status(101).with_stderr(
            "\
[UPDATING] registry [..]
[UPDATING] git repository [..]
error: no matching package for override `[..]baz:0.1.0` found
location searched: file://[..]
version required: = 0.1.0
",
        ),
    );
}

#[test]
fn override_with_nothing() {
    Package::new("bar", "0.1.0").publish();

    let foo = git::repo(&paths::root().join("override"))
        .file("src/lib.rs", "")
        .build();

    let p = project()
        .file(
            "Cargo.toml",
            &format!(
                r#"
            [package]
            name = "foo"
            version = "0.0.1"
            authors = []

            [dependencies]
            bar = "0.1"

            [replace]
            "bar:0.1.0" = {{ git = '{}' }}
        "#,
                foo.url()
            ),
        )
        .file("src/lib.rs", "")
        .build();

    assert_that(
        p.cargo("build"),
        execs().with_status(101).with_stderr(
            "\
[UPDATING] registry [..]
[UPDATING] git repository [..]
[ERROR] failed to load source for a dependency on `bar`

Caused by:
  Unable to update file://[..]

Caused by:
  Could not find Cargo.toml in `[..]`
",
        ),
    );
}

#[test]
fn override_wrong_version() {
    let p = project()
        .file(
            "Cargo.toml",
            r#"
            [package]
            name = "foo"
            version = "0.0.1"
            authors = []

            [replace]
            "bar:0.1.0" = { git = 'https://example.com', version = '0.2.0' }
        "#,
        )
        .file("src/lib.rs", "")
        .build();

    assert_that(
        p.cargo("build"),
        execs().with_status(101).with_stderr(
            "\
error: failed to parse manifest at `[..]`

Caused by:
  replacements cannot specify a version requirement, but found one for `[..]bar:0.1.0`
",
        ),
    );
}

#[test]
fn multiple_specs() {
    Package::new("bar", "0.1.0").publish();

    let bar = git::repo(&paths::root().join("override"))
        .file("Cargo.toml", &basic_manifest("bar", "0.1.0"))
        .file("src/lib.rs", "pub fn bar() {}")
        .build();

    let p = project()
        .file(
            "Cargo.toml",
            &format!(
                r#"
            [package]
            name = "foo"
            version = "0.0.1"
            authors = []

            [dependencies]
            bar = "0.1.0"

            [replace]
            "bar:0.1.0" = {{ git = '{0}' }}

            [replace."https://github.com/rust-lang/crates.io-index#bar:0.1.0"]
            git = '{0}'
        "#,
                bar.url()
            ),
        )
        .file("src/lib.rs", "")
        .build();

    assert_that(
        p.cargo("build"),
        execs().with_status(101).with_stderr(
            "\
[UPDATING] registry [..]
[UPDATING] git repository [..]
error: overlapping replacement specifications found:

  * [..]
  * [..]

both specifications match: bar v0.1.0
",
        ),
    );
}

#[test]
fn test_override_dep() {
    Package::new("bar", "0.1.0").publish();

    let bar = git::repo(&paths::root().join("override"))
        .file("Cargo.toml", &basic_manifest("bar", "0.1.0"))
        .file("src/lib.rs", "pub fn bar() {}")
        .build();

    let p = project()
        .file(
            "Cargo.toml",
            &format!(
                r#"
            [package]
            name = "foo"
            version = "0.0.1"
            authors = []

            [dependencies]
            bar = "0.1.0"

            [replace]
            "bar:0.1.0" = {{ git = '{0}' }}
        "#,
                bar.url()
            ),
        )
        .file("src/lib.rs", "")
        .build();

    assert_that(
        p.cargo("test").arg("-p").arg("bar"),
        execs().with_status(101).with_stderr_contains(
            "\
error: There are multiple `bar` packages in your project, and the [..]
Please re-run this command with [..]
  [..]#bar:0.1.0
  [..]#bar:0.1.0
",
        ),
    );
}

#[test]
fn update() {
    Package::new("bar", "0.1.0").publish();

    let bar = git::repo(&paths::root().join("override"))
        .file("Cargo.toml", &basic_manifest("bar", "0.1.0"))
        .file("src/lib.rs", "pub fn bar() {}")
        .build();

    let p = project()
        .file(
            "Cargo.toml",
            &format!(
                r#"
            [package]
            name = "foo"
            version = "0.0.1"
            authors = []

            [dependencies]
            bar = "0.1.0"

            [replace]
            "bar:0.1.0" = {{ git = '{0}' }}
        "#,
                bar.url()
            ),
        )
        .file("src/lib.rs", "")
        .build();

    assert_that(p.cargo("generate-lockfile"), execs());
    assert_that(
        p.cargo("update"),
        execs().with_stderr(
            "\
[UPDATING] registry `[..]`
[UPDATING] git repository `[..]`
",
        ),
    );
}

// foo -> near -> far
// near is overridden with itself
#[test]
fn no_override_self() {
    let deps = git::repo(&paths::root().join("override"))
        .file("far/Cargo.toml", &basic_manifest("far", "0.1.0"))
        .file("far/src/lib.rs", "")
        .file(
            "near/Cargo.toml",
            r#"
            [package]
            name = "near"
            version = "0.1.0"
            authors = []

            [dependencies]
            far = { path = "../far" }
        "#,
        )
        .file("near/src/lib.rs", "#![no_std] pub extern crate far;")
        .build();

    let p = project()
        .file(
            "Cargo.toml",
            &format!(
                r#"
            [package]
            name = "foo"
            version = "0.0.1"
            authors = []

            [dependencies]
            near = {{ git = '{0}' }}

            [replace]
            "near:0.1.0" = {{ git = '{0}' }}
        "#,
                deps.url()
            ),
        )
        .file("src/lib.rs", "#![no_std] pub extern crate near;")
        .build();

    assert_that(p.cargo("build").arg("--verbose"), execs());
}

#[test]
fn broken_path_override_warns() {
    Package::new("bar", "0.1.0").publish();
    Package::new("bar", "0.2.0").publish();

    let p = project()
        .file(
            "Cargo.toml",
            r#"
            [package]
            name = "foo"
            version = "0.0.1"
            authors = []

            [dependencies]
            a = { path = "a1" }
        "#,
        )
        .file("src/lib.rs", "")
        .file(
            "a1/Cargo.toml",
            r#"
            [package]
            name = "a"
            version = "0.0.1"
            authors = []

            [dependencies]
            bar = "0.1"
        "#,
        )
        .file("a1/src/lib.rs", "")
        .file(
            "a2/Cargo.toml",
            r#"
            [package]
            name = "a"
            version = "0.0.1"
            authors = []

            [dependencies]
            bar = "0.2"
        "#,
        )
        .file("a2/src/lib.rs", "")
        .file(".cargo/config", r#"paths = ["a2"]"#)
        .build();

    assert_that(
        p.cargo("build"),
        execs().with_stderr(
            "\
[UPDATING] [..]
warning: path override for crate `a` has altered the original list of
dependencies; the dependency on `bar` was either added or
modified to not match the previously resolved version

This is currently allowed but is known to produce buggy behavior with spurious
recompiles and changes to the crate graph. Path overrides unfortunately were
never intended to support this feature, so for now this message is just a
warning. In the future, however, this message will become a hard error.

To change the dependency graph via an override it's recommended to use the
`[replace]` feature of Cargo instead of the path override feature. This is
documented online at the url below for more information.

http://doc.crates.io/specifying-dependencies.html#overriding-dependencies

[DOWNLOADING] [..]
[COMPILING] [..]
[COMPILING] [..]
[COMPILING] [..]
[FINISHED] [..]
",
        ),
    );
}

#[test]
fn override_an_override() {
    Package::new("chrono", "0.2.0")
        .dep("serde", "< 0.9")
        .publish();
    Package::new("serde", "0.7.0")
        .file("src/lib.rs", "pub fn serde07() {}")
        .publish();
    Package::new("serde", "0.8.0")
        .file("src/lib.rs", "pub fn serde08() {}")
        .publish();

    let p = project()
        .file(
            "Cargo.toml",
            r#"
            [package]
            name = "foo"
            version = "0.0.1"
            authors = []

            [dependencies]
            chrono = "0.2"
            serde = "0.8"

            [replace]
            "chrono:0.2.0" = { path = "chrono" }
            "serde:0.8.0" = { path = "serde" }
        "#,
        )
        .file(
            "Cargo.lock",
            r#"
            [[package]]
            name = "foo"
            version = "0.0.1"
            dependencies = [
             "chrono 0.2.0 (registry+https://github.com/rust-lang/crates.io-index)",
             "serde 0.8.0 (registry+https://github.com/rust-lang/crates.io-index)",
            ]

            [[package]]
            name = "chrono"
            version = "0.2.0"
            source = "registry+https://github.com/rust-lang/crates.io-index"
            replace = "chrono 0.2.0"

            [[package]]
            name = "chrono"
            version = "0.2.0"
            dependencies = [
             "serde 0.7.0 (registry+https://github.com/rust-lang/crates.io-index)",
            ]

            [[package]]
            name = "serde"
            version = "0.7.0"
            source = "registry+https://github.com/rust-lang/crates.io-index"

            [[package]]
            name = "serde"
            version = "0.8.0"
            source = "registry+https://github.com/rust-lang/crates.io-index"
            replace = "serde 0.8.0"

            [[package]]
            name = "serde"
            version = "0.8.0"
        "#,
        )
        .file(
            "src/lib.rs",
            "
            extern crate chrono;
            extern crate serde;

            pub fn foo() {
                chrono::chrono();
                serde::serde08_override();
            }
        ",
        )
        .file(
            "chrono/Cargo.toml",
            r#"
            [package]
            name = "chrono"
            version = "0.2.0"
            authors = []

            [dependencies]
            serde = "< 0.9"
        "#,
        )
        .file(
            "chrono/src/lib.rs",
            "
            extern crate serde;
            pub fn chrono() {
                serde::serde07();
            }
        ",
        )
        .file("serde/Cargo.toml", &basic_manifest("serde", "0.8.0"))
        .file("serde/src/lib.rs", "pub fn serde08_override() {}")
        .build();

    assert_that(p.cargo("build").arg("-v"), execs());
}

#[test]
fn overriding_nonexistent_no_spurious() {
    Package::new("bar", "0.1.0").dep("baz", "0.1").publish();
    Package::new("baz", "0.1.0").publish();

    let bar = git::repo(&paths::root().join("override"))
        .file(
            "Cargo.toml",
            r#"
            [package]
            name = "bar"
            version = "0.1.0"
            authors = []

            [dependencies]
            baz = { path = "baz" }
        "#,
        )
        .file("src/lib.rs", "pub fn bar() {}")
        .file("baz/Cargo.toml", &basic_manifest("baz", "0.1.0"))
        .file("baz/src/lib.rs", "pub fn baz() {}")
        .build();

    let p = project()
        .file(
            "Cargo.toml",
            &format!(
                r#"
            [package]
            name = "foo"
            version = "0.0.1"
            authors = []

            [dependencies]
            bar = "0.1.0"

            [replace]
            "bar:0.1.0" = {{ git = '{url}' }}
            "baz:0.1.0" = {{ git = '{url}' }}
        "#,
                url = bar.url()
            ),
        )
        .file("src/lib.rs", "")
        .build();

    assert_that(p.cargo("build"), execs());
    assert_that(
        p.cargo("build"),
        execs()
            .with_stderr(
                "\
[WARNING] package replacement is not used: [..]baz:0.1.0
[FINISHED] [..]
",
            )
            .with_stdout(""),
    );
}

#[test]
fn no_warnings_when_replace_is_used_in_another_workspace_member() {
    Package::new("bar", "0.1.0").publish();
    Package::new("baz", "0.1.0").publish();

    let p = project()
        .file(
            "Cargo.toml",
            r#"
            [workspace]
            members = [ "first_crate", "second_crate"]

            [replace]
            "bar:0.1.0" = { path = "local_bar" }"#,
        )
        .file(
            "first_crate/Cargo.toml",
            r#"
            [package]
            name = "first_crate"
            version = "0.1.0"

            [dependencies]
            bar = "0.1.0"
        "#,
        )
        .file("first_crate/src/lib.rs", "")
        .file("second_crate/Cargo.toml", &basic_manifest("second_crate", "0.1.0"))
        .file("second_crate/src/lib.rs", "")
        .file("local_bar/Cargo.toml", &basic_manifest("bar", "0.1.0"))
        .file("local_bar/src/lib.rs", "")
        .build();

    assert_that(
        p.cargo("build").cwd(p.root().join("first_crate")),
        execs().with_stdout("").with_stderr(
            "\
[UPDATING] registry `[..]`
[COMPILING] bar v0.1.0 ([..])
[COMPILING] first_crate v0.1.0 ([..])
[FINISHED] [..]",
        ),
    );

    assert_that(
        p.cargo("build").cwd(p.root().join("second_crate")),
        execs().with_stdout("").with_stderr(
            "\
[COMPILING] second_crate v0.1.0 ([..])
[FINISHED] [..]",
        ),
    );
}

#[test]
fn override_to_path_dep() {
    Package::new("bar", "0.1.0").dep("baz", "0.1").publish();
    Package::new("baz", "0.1.0").publish();

    let p = project()
        .file(
            "Cargo.toml",
            r#"
            [package]
            name = "foo"
            version = "0.0.1"
            authors = []

            [dependencies]
            bar = "0.1.0"
        "#,
        )
        .file("src/lib.rs", "")
        .file(
            "bar/Cargo.toml",
            r#"
            [package]
            name = "bar"
            version = "0.0.1"
            authors = []

            [dependencies]
            baz = { path = "baz" }
        "#,
        )
        .file("bar/src/lib.rs", "")
        .file("bar/baz/Cargo.toml", &basic_manifest("baz", "0.0.1"))
        .file("bar/baz/src/lib.rs", "")
        .file(".cargo/config", r#"paths = ["bar"]"#)
        .build();

    assert_that(p.cargo("build"), execs());
}

#[test]
fn replace_to_path_dep() {
    Package::new("bar", "0.1.0").dep("baz", "0.1").publish();
    Package::new("baz", "0.1.0").publish();

    let p = project()
        .file(
            "Cargo.toml",
            r#"
            [package]
            name = "foo"
            version = "0.0.1"
            authors = []

            [dependencies]
            bar = "0.1.0"

            [replace]
            "bar:0.1.0" = { path = "bar" }
        "#,
        )
        .file("src/lib.rs", "extern crate bar;")
        .file(
            "bar/Cargo.toml",
            r#"
            [package]
            name = "bar"
            version = "0.1.0"
            authors = []

            [dependencies]
            baz = { path = "baz" }
        "#,
        )
        .file("bar/src/lib.rs", "extern crate baz; pub fn bar() { baz::baz(); }")
        .file("bar/baz/Cargo.toml", &basic_manifest("baz", "0.1.0"))
        .file("bar/baz/src/lib.rs", "pub fn baz() {}")
        .build();

    assert_that(p.cargo("build"), execs());
}

#[test]
fn paths_ok_with_optional() {
    Package::new("baz", "0.1.0").publish();

    let p = project()
        .file(
            "Cargo.toml",
            r#"
            [package]
            name = "foo"
            version = "0.0.1"
            authors = []

            [dependencies]
            bar = { path = "bar" }
        "#,
        )
        .file("src/lib.rs", "")
        .file(
            "bar/Cargo.toml",
            r#"
            [package]
            name = "bar"
            version = "0.1.0"
            authors = []

            [dependencies]
            baz = { version = "0.1", optional = true }
        "#,
        )
        .file("bar/src/lib.rs", "")
        .file(
            "bar2/Cargo.toml",
            r#"
            [package]
            name = "bar"
            version = "0.1.0"
            authors = []

            [dependencies]
            baz = { version = "0.1", optional = true }
        "#,
        )
        .file("bar2/src/lib.rs", "")
        .file(".cargo/config", r#"paths = ["bar2"]"#)
        .build();

    assert_that(
        p.cargo("build"),
        execs().with_stderr(
            "\
[COMPILING] bar v0.1.0 ([..]bar2)
[COMPILING] foo v0.0.1 ([..])
[FINISHED] [..]
",
        ),
    );
}

#[test]
fn paths_add_optional_bad() {
    Package::new("baz", "0.1.0").publish();

    let p = project()
        .file(
            "Cargo.toml",
            r#"
            [package]
            name = "foo"
            version = "0.0.1"
            authors = []

            [dependencies]
            bar = { path = "bar" }
        "#,
        )
        .file("src/lib.rs", "")
        .file("bar/Cargo.toml", &basic_manifest("bar", "0.1.0"))
        .file("bar/src/lib.rs", "")
        .file(
            "bar2/Cargo.toml",
            r#"
            [package]
            name = "bar"
            version = "0.1.0"
            authors = []

            [dependencies]
            baz = { version = "0.1", optional = true }
        "#,
        )
        .file("bar2/src/lib.rs", "")
        .file(".cargo/config", r#"paths = ["bar2"]"#)
        .build();

    assert_that(
        p.cargo("build"),
        execs().with_stderr_contains(
            "\
warning: path override for crate `bar` has altered the original list of
dependencies; the dependency on `baz` was either added or\
",
        ),
    );
}

#[test]
fn override_with_default_feature() {
    Package::new("another", "0.1.0").publish();
    Package::new("another", "0.1.1").dep("bar", "0.1").publish();
    Package::new("bar", "0.1.0").publish();

    let p = project()
        .file(
            "Cargo.toml",
            r#"
            [package]
            name = "foo"
            version = "0.0.1"
            authors = []

            [dependencies]
            bar = { path = "bar", default-features = false }
            another = "0.1"
            another2 = { path = "another2" }

            [replace]
            'bar:0.1.0' = { path = "bar" }
        "#,
        )
        .file("src/main.rs", "extern crate bar; fn main() { bar::bar(); }")
        .file(
            "bar/Cargo.toml",
            r#"
            [package]
            name = "bar"
            version = "0.1.0"
            authors = []

            [features]
            default = []
        "#,
        )
        .file(
            "bar/src/lib.rs",
            r#"
            #[cfg(feature = "default")]
            pub fn bar() {}
        "#,
        )
        .file(
            "another2/Cargo.toml",
            r#"
            [package]
            name = "another2"
            version = "0.1.0"
            authors = []

            [dependencies]
            bar = { version = "0.1", default-features = false }
        "#,
        )
        .file("another2/src/lib.rs", "")
        .build();

    assert_that(p.cargo("run"), execs());
}

#[test]
fn override_plus_dep() {
    Package::new("bar", "0.1.0").publish();

    let p = project()
        .file(
            "Cargo.toml",
            r#"
            [package]
            name = "foo"
            version = "0.0.1"
            authors = []

            [dependencies]
            bar = "0.1"

            [replace]
            'bar:0.1.0' = { path = "bar" }
        "#,
        )
        .file("src/lib.rs", "")
        .file(
            "bar/Cargo.toml",
            r#"
            [package]
            name = "bar"
            version = "0.1.0"
            authors = []

            [dependencies]
            foo = { path = ".." }
        "#,
        )
        .file("bar/src/lib.rs", "")
        .build();

    assert_that(
        p.cargo("build"),
        execs()
            .with_status(101)
            .with_stderr_contains("error: cyclic package dependency: [..]"),
    );
}
